<?php
namespace Hostiko_Toolkit_Addon;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

 /**
 * For Elementor Global Kit Settings
 */
use Elementor\Core\App\Modules\ImportExport\Directories\Root;
use Elementor\Core\App\Modules\ImportExport\Module;
use Elementor\Core\Settings\Page\Manager as PageManager;
use Elementor\Plugin;


/**
 * Hostiko_Plugin class.
 *
 * The main class that initiates and runs the addon.
 *
 * @since 1.0.0
 */
final class Hostiko_Plugin {

	/**
	 * Addon Version
	 *
	 * @since 1.0.0
	 * @var string The addon version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 * @var string Minimum Elementor version required to run the addon.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '3.7.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 * @var string Minimum PHP version required to run the addon.
	 */
	const MINIMUM_PHP_VERSION = '7.3';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access private
	 * @static
	 * @var \Hostiko_Toolkit_Addon\Hostiko_Plugin The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 * @return \Hostiko_Toolkit_Addon\Hostiko_Plugin An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * Perform some compatibility checks to make sure basic requirements are meet.
	 * If all compatibility checks pass, initialize the functionality.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {

		if ( $this->is_compatible() ) {
			add_action( 'elementor/init', [ $this, 'init' ] );
		}
		$this->register_callbacks();
	}

	/**
	 * Compatibility Checks
	 *
	 * Checks whether the site meets the addon requirement.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function is_compatible() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return false;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return false;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return false;
		}

		return true;

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'hostiko-toolkit-addon' ),
			'<strong>' . esc_html__( 'Hostiko Toolkit Addon', 'hostiko-toolkit-addon' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'hostiko-toolkit-addon' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'hostiko-toolkit-addon' ),
			'<strong>' . esc_html__( 'Hostiko Toolkit Addon', 'hostiko-toolkit-addon' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'hostiko-toolkit-addon' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'hostiko-toolkit-addon' ),
			'<strong>' . esc_html__( 'Hostiko Toolkit Addon', 'hostiko-toolkit-addon' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'hostiko-toolkit-addon' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}


	/**
	 * Initialize
	 *
	 * Load the addons functionality only after Elementor is initialized.
	 *
	 * Fired by `elementor/init` action hook.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function init() {

		add_action( 'elementor/widgets/register', [ $this, 'register_widgets' ] );
		add_action( 'elementor/controls/register', [ $this, 'register_controls' ] );

	}

	/**
	 * Register Widgets
	 *
	 * Load widgets files and register new Elementor widgets.
	 *
	 * Fired by `elementor/widgets/register` action hook.
	 *
	 * @param \Elementor\Widgets_Manager $widgets_manager Elementor widgets manager.
	 */
	public function register_widgets( $widgets_manager ) {

		require_once( __DIR__ . '/widgets/hostiko-pricing-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-testimonial-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-tld-table-widget.php' );
		require_once( __DIR__ . '/widgets/whmcs-product-name-widget.php' );
		require_once( __DIR__ . '/widgets/whmcs-product-description-widget.php' );
		require_once( __DIR__ . '/widgets/whmcs-product-price-widget.php' );
		require_once( __DIR__ . '/widgets/whmcs-price-plan-01-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-vps-widget.php' );
		require_once( __DIR__ . '/widgets/whmcs-domain-search-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-tabs-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-shared-table-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-dedicated-table-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-dedicated-server-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-hosting-feature-widget.php' );
		require_once( __DIR__ . '/widgets/whmcs-product-button-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-blog-post-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-blog-archive-widget.php' );
		require_once( __DIR__ . '/widgets/hostiko-data-centers-widget.php' );

		$widgets_manager->register( new \Elementor_Hostiko_Pricing_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Testimonial_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Tld_Table_Widget() );
		$widgets_manager->register( new \Elementor_Whmcs_Product_Widget() );
		$widgets_manager->register( new \Elementor_Whmcs_Product_Description_Widget() );
		$widgets_manager->register( new \Elementor_Whmcs_Product_Price_Widget() );
		$widgets_manager->register( new \Elementor_Whmcs_Price_Plan_01_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Vps_Widget() );
		$widgets_manager->register( new \Elementor_Whmcs_Domain_Search_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Tabs_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Shared_Table_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Dedicated_Table_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Dedicated_Server_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Hosting_Feature_Widget() );
		$widgets_manager->register( new \Elementor_Whmcs_Product_Button_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Blog_Post_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Blog_Archive_Widget() );
		$widgets_manager->register( new \Elementor_Hostiko_Data_Centers_Widget() );
		
	}

	/**
	 * Register Controls
	 *
	 * Load controls files and register new Elementor controls.
	 *
	 * Fired by `elementor/controls/register` action hook.
	 *
	 * @param \Elementor\Controls_Manager $controls_manager Elementor controls manager.
	 */
	public function register_controls( $controls_manager ) {

		require_once( __DIR__ . '/controls/hostiko-pricing-control.php' );
		require_once( __DIR__ . '/controls/hostiko-testimonial-control.php' );
		require_once( __DIR__ . '/controls/hostiko-tld-table-control.php' );
		require_once( __DIR__ . '/controls/whmcs-product-name-control.php' );
		require_once( __DIR__ . '/controls/whmcs-product-description-control.php' );
		require_once( __DIR__ . '/controls/whmcs-product-price-control.php' );
		require_once( __DIR__ . '/controls/whmcs-price-plan-01-control.php' );
		require_once( __DIR__ . '/controls/hostiko-vps-control.php' );
		require_once( __DIR__ . '/controls/whmcs-domain-search-control.php' );
		require_once( __DIR__ . '/controls/hostiko-tabs-control.php' );
		require_once( __DIR__ . '/controls/hostiko-shared-table-control.php' );
		require_once( __DIR__ . '/controls/hostiko-dedicated-table-control.php' );
		require_once( __DIR__ . '/controls/hostiko-dedicated-server-control.php' );
		require_once( __DIR__ . '/controls/hostiko-hosting-feature-control.php' );
		require_once( __DIR__ . '/controls/whmcs-product-button-control.php' );
		require_once( __DIR__ . '/controls/hostiko-blog-post-control.php' );
		require_once( __DIR__ . '/controls/hostiko-blog-archive-control.php' );
		require_once( __DIR__ . '/controls/hostiko-data-centers-control.php' );

		$controls_manager->register( new \Elementor_Hostiko_Pricing_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Testimonial_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Tld_Table_Control() );
		$controls_manager->register( new \Elementor_Whmcs_Product_Name_Control() );
		$controls_manager->register( new \Elementor_Whmcs_Product_Description_Control() );
		$controls_manager->register( new \Elementor_Whmcs_Product_Price_Control() );
		$controls_manager->register( new \Elementor_Whmcs_Price_Plan_01_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Vps_Control() );
		$controls_manager->register( new \Elementor_Whmcs_Domain_Search_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Tabs_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Shared_Table_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Dedicated_Table_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Dedicated_Server_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Hosting_Feature_Control() );
		$controls_manager->register( new \Elementor_Whmcs_Product_Button_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Blog_Post_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Blog_Archive_Control() );
		$controls_manager->register( new \Elementor_Hostiko_Data_Centers_Control() );

	}
	protected function register_callbacks(){
        add_filter('akd_elementor_global', array($this, 'akd_elementor_global'));
    }

	public function akd_elementor_global(){
		$data = get_transient( 'ocdi_importer_data' );
        $kit = Plugin::$instance
            ->kits_manager
            ->get_active_kit();
        $old_settings = $kit->get_meta(PageManager::META_KEY);
        if (!$old_settings){
        	$old_settings = [];
        }
        //$new_settings = json_decode(file_get_contents(get_template_directory() . '/demo/layout-01/site-settings.json') , true);
        $new_settings = json_decode(file_get_contents($data['selected_import_files']['widgets']) , true);
        $new_settings = $new_settings['settings'];
        if ($old_settings){
            if (isset($old_settings['custom_colors'])){
                $new_settings['custom_colors'] = array_merge($old_settings['custom_colors'], $new_settings['custom_colors']);
            }
            if (isset($old_settings['custom_typography'])){
                $new_settings['custom_typography'] = array_merge($old_settings['custom_typography'], $new_settings['custom_typography']);
            }
        }
        $new_settings = array_replace_recursive($old_settings, $new_settings);
        $kit->save(['settings' => $new_settings]);

		
		// Import WP Options

		// $data = get_transient( 'ocdi_importer_data' );
		$layout_no = $data['selected_index'];
	   
		// Specify the path to your JSON file
		$json_file_path = $data['import_files'][$layout_no]['local_import_options_file'];
		//$json_file_path = ABSPATH . 'wp-content/themes/hostiko/demo/layout-01/exported-options.json';
	  
	
		// Read the JSON file
		$json_data = file_get_contents($json_file_path);
	
		// Decode JSON data
		$imported_options = json_decode($json_data, true);
	
		// Check if options exist
		if ($imported_options && is_array($imported_options)) {
			// Loop through each option
			foreach ($imported_options as $option_name => $option_value) {
				// Update or add the option in the WordPress options table
				update_option($option_name, $option_value);
			}
		}
    }
}


