<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://http://designingmedia.com/
 * @since      1.0.0
 *
 * @package    Akd_Framework
 * @subpackage Akd_Framework/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Akd_Framework
 * @subpackage Akd_Framework/includes
 * @author     Ammar Nasir <info@domain.com>
 */
class Akd_Framework {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Akd_Framework_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'AKD_FRAMEWORK_VERSION' ) ) {
			$this->version = AKD_FRAMEWORK_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'akd-framework';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Akd_Framework_Loader. Orchestrates the hooks of the plugin.
	 * - Akd_Framework_i18n. Defines internationalization functionality.
	 * - Akd_Framework_Admin. Defines all hooks for the admin area.
	 * - Akd_Framework_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/merlin/vendor/autoload.php';

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-akd-framework-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-akd-framework-i18n.php';

		/**
		 * Include merlin framework
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/merlin/class-merlin.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/merlin/merlin-config.php';

		/**
		 * The class responsible for handling activation of deactivation process of theme
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-akd-theme-activation.php';

		/**
		 * The class responsible for handling theme's required plugins installation
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-akd-required-plugins-activation.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-akd-framework-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-akd-framework-public.php';

		/**
		 * Class responsible for creating meta fields
		 */
		if ( class_exists( 'CSF' ) ) {

			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-akd-framework-meta.php';
			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/meta-boxes/class-akd-framework-admin-meta.php';
			
			new Akd_Framework_Admin_Meta( $this->plugin_name );
		}

		$this->loader = new Akd_Framework_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Akd_Framework_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new Akd_Framework_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Akd_Framework_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		//AJAX request validate purchase code
		$this->loader->add_action( 'wp_ajax_validate_purchase_code', $plugin_admin, 'validate_purchase_code_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_validate_purchase_code', $plugin_admin, 'validate_purchase_code_handler' );

		//AJAX request to disconnect domain
		$this->loader->add_action( 'wp_ajax_disconnect_domain', $plugin_admin, 'disconnect_domain_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_disconnect_domain', $plugin_admin, 'disconnect_domain_handler' );

		//AJAX request to install child theme
		$this->loader->add_action( 'wp_ajax_install_child_theme', $plugin_admin, 'install_child_theme_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_install_child_theme', $plugin_admin, 'install_child_theme_handler' );

		//AJAX request to download theme files
		$this->loader->add_action( 'wp_ajax_download_theme_files', $plugin_admin, 'download_theme_files_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_download_theme_files', $plugin_admin, 'download_theme_files_handler' );

		//AJAX request to download import files
		$this->loader->add_action( 'wp_ajax_download_import_files', $plugin_admin, 'download_import_files_handler' );
		$this->loader->add_action( 'wp_ajax_nopriv_download_import_files', $plugin_admin, 'download_import_files_handler' );

		//AJAX request to check whmcs curl
		$this->loader->add_action( 'wp_ajax_checkCurl', $plugin_admin, 'checkCurl' );
		$this->loader->add_action( 'wp_ajax_nopriv_checkCurl', $plugin_admin, 'checkCurl' );

		//AJAX request to get whmcs groups
		$this->loader->add_action( 'wp_ajax_getGroups', $plugin_admin, 'getGroups' );
		$this->loader->add_action( 'wp_ajax_nopriv_getGroups', $plugin_admin, 'getGroups' );

		//AJAX request to get whmcs tlds
		$this->loader->add_action( 'wp_ajax_getTLDs', $plugin_admin, 'getTLDs' );
		$this->loader->add_action( 'wp_ajax_nopriv_getTLDs', $plugin_admin, 'getTLDs' );

		//AJAX request to resync whmcs
		$this->loader->add_action( 'wp_ajax_getGroups_resync', $plugin_admin, 'getGroups_resync' );
		$this->loader->add_action( 'wp_ajax_nopriv_getGroups_resync', $plugin_admin, 'getGroups_resync' );
		

		//trigger melin framework
		$this->loader->add_action( 'admin_init', $plugin_admin, 'akd_admin_init_callback' );

		//hook to import demo content via merlin framework
		$this->loader->add_filter( 'merlin_import_files', $plugin_admin, 'akd_import_theme_demo_files' );

		//hook to list theme's required plugins
		$this->loader->add_action( 'tgmpa_register', $plugin_admin, 'akd_register_required_plugins' );

		//hook to validate data and creation of WHMCS-sync page
		// $this->loader->add_action( "csf_akd-framework-admin-page_save_before", $plugin_admin, 'akd_update_elementor_global_settings' );

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Akd_Framework_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Akd_Framework_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
