<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://http://designingmedia.com/
 * @since      1.0.0
 *
 * @package    Akd_Framework
 * @subpackage Akd_Framework/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Akd_Framework
 * @subpackage Akd_Framework/admin
 * @author     Ammar Nasir <info@domain.com>
 */
class Akd_Framework_Admin {

	use Akd_Theme_Activation;
	use Akd_Required_plugin_Activation;

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	private $activated_theme;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->activated_theme = wp_get_theme();

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Akd_Framework_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Akd_Framework_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/akd-framework-admin.css', array(), $this->version, 'all' );
		wp_enqueue_style( 'sweetalert', 'https://cdnjs.cloudflare.com/ajax/libs/sweetalert2/11.11.0/sweetalert2.min.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Akd_Framework_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Akd_Framework_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/akd-framework-admin.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'merlin', plugins_url() .'/akd-demo-importer/includes/merlin/assets/js/merlin.js', array( 'jquery' ), $this->version );
		wp_enqueue_script( 'sweetalert', 'https://cdnjs.cloudflare.com/ajax/libs/sweetalert2/11.11.0/sweetalert2.min.js', array( 'jquery' ), $this->version, false );

		// Localize the script with URLs
		$urls = [
			'siteUrl' => site_url(),
			'ajaxUrl' => admin_url('admin-ajax.php'),
		];

		$elementor = false;
		// Check if Elementor plugin is active
		if (in_array('elementor/elementor.php', apply_filters('active_plugins', get_option('active_plugins')))) {
			$elementor = true;
		}

		$additional_data = [];
		if(get_option('_pruchase_code') && get_option('_encryption_key')){
			$additional_data = [
				'purchaseCode' => $this->akd_decrypt_data( get_option('_pruchase_code'), get_option('_encryption_key') ),
			];
		}

		// Check if TGMPA is loaded (optional)
		if (class_exists('TGM_Plugin_Activation')) {
			// Check if TGMPA is initialized
			if (isset($GLOBALS['tgmpa'])) {
				// Get TGMPA instance
				$tgmpa_instance = $GLOBALS['tgmpa'];
				$required_plugins = '';
				if(TGM_Plugin_Activation::get_instance()->is_tgmpa_complete() == true){
					$required_plugins = 'installed';
				}
			}
		}

		$child_theme_directory = get_theme_root() . '/hostiko-child'; // Path to the child theme director

		$theme_info = [
			'themeName' => $this->activated_theme->get('Name'),
			'childTheme' => $this->activated_theme->get('Template'),
			'elementor' => $elementor,
			'requiredPlugins' => $required_plugins,
			'childThemeFound' => is_dir($child_theme_directory)
		];

		global $wpdb;

		// Create an array to store all the information
		$server_info = [
			'wp_version' 			 => get_bloginfo('version'),
			'wp_dir_writeable' 		 => is_writable(wp_upload_dir()['basedir']),
			'php_v' 				 => phpversion(),
			'php_memory_limit' 		 => ini_get('memory_limit'),
			'php_post_max_size' 	 => ini_get('post_max_size'),
			'php_max_upload_size' 	 => ini_get('upload_max_filesize'),
			'php_max_input_vars' 	 => ini_get('max_input_vars'),
			'php_max_execution_time' => ini_get('max_execution_time'),
			'wp_memory_limit' 		 => WP_MEMORY_LIMIT,
			'mysql_v' 				 => $wpdb->db_version()
		];


		// Combine localization data into a single array
		$localization_data = [
			'urls_to_frontend'    => $urls,
			'additional_data'	  => $additional_data,
			'theme_info'		  => $theme_info,
			'server_info'		  => $server_info,
			'theme_demo' 		  => get_option('_akd_demo_import') ? get_option('_akd_demo_import') : '',
			'valid_purchase_code' => get_option('_purchase_code_validate') ? get_option('_purchase_code_validate') : '',
			'whmcs_url'			  => get_option('whmcs_url'),
			'whmcs_identifier'	  => get_option('whmcs_identifier'),
			'whmcs_secret'	  	  => get_option('whmcs_secret'),
		];

		// Localize the script with combined data
		wp_localize_script($this->plugin_name, 'akd_localization_data', $localization_data);

		wp_localize_script(
			'merlin', 'merlin_params', array(
				'tgm_plugin_nonce' => array(
					'update'  => wp_create_nonce( 'tgmpa-update' ),
					'install' => wp_create_nonce( 'tgmpa-install' ),
				),
				'ajaxurl'          => admin_url( 'admin-ajax.php' ),
				'wpnonce'          => wp_create_nonce( 'merlin_nonce' ),
			)
		);

	}

	public function pre($data){
		echo '<pre>'; print_r($data);
	}

	/**
	 * Method to trigger admin_init hook methods
	 */
	public function akd_admin_init_callback(){
		//method to remove GDPR cookie compliance warnings from the merlin page
		$this->akd_remove_cookie_compliance_warnings();

		//method to get and save elementor global settings within wp_options
		$this->akd_get_elementor_global_settings();

		//method to process manual import
		$this->process_manual_import();

		if(get_option('_akd_demo_import')){
			$this->akd_delete_layout_folders_after_import();
		}

	}

	// Method to check and delete layout-* folders
    public function akd_delete_layout_folders_after_import() {
        // Get the uploads directory path
        $uploads_dir = wp_upload_dir()['path'];

        // Check if the directory exists
        if (is_dir($uploads_dir)) {
            // Scan the directory for subfolders matching the pattern 'layout-*'
            $subfolders = array_filter(glob($uploads_dir . '/layout-*'), 'is_dir');

            // Loop through each found subfolder and delete it
            foreach ($subfolders as $folder) {
                if ($this->delete_dir($folder)) {
                    // echo 'Deleted folder: ' . basename($folder) . '<br>';
                } else {
                    // echo 'Failed to delete folder: ' . basename($folder) . '<br>';
                }
            }
        } else {
            // echo 'Uploads directory does not exist.';
        }
    }


	/**
	 * Method to remove files from the dir
	 */
	private function delete_dir($dir) {
		if (!file_exists($dir)) {
			return true;
		}
	
		if (!is_dir($dir) || is_link($dir)) {
			return unlink($dir);
		}
	
		foreach (scandir($dir) as $item) {
			if ($item == '.' || $item == '..') {
				continue;
			}
	
			if (!$this->delete_dir($dir . DIRECTORY_SEPARATOR . $item)) {
				return false;
			}
	
		}
	
		return rmdir($dir);
	}

	/**
	 * Method to process manual import
	 */
	private function process_manual_import(){
		$demo_directory = plugin_dir_path(dirname(__FILE__)) . 'admin/demo/';
		$zip_files = glob($demo_directory . '*.zip');

		if (!empty($zip_files)) {
			foreach ($zip_files as $zip_file) {
				
				// Extract the folder name from the zip file name
				$folder_name = pathinfo($zip_file, PATHINFO_FILENAME);

				if (strpos($folder_name, 'manual-') === 0) {
					// Remove the "manual-" prefix
					$folder_name = substr($folder_name, 7);
					$new_zip_file_name = $folder_name . '.zip';
					$new_zip_file_path = $demo_directory . $new_zip_file_name;
	
					// Rename the zip file
					rename($zip_file, $new_zip_file_path);
				}

				// Check if the folder exists, and remove it
				$existing_folder = $demo_directory . $folder_name;
				if (is_dir($existing_folder)) {
					// Remove existing folder and its contents
					$this->delete_dir($existing_folder);
				}

				$zip_file = $new_zip_file_path;

				// Unzip the file
				$zip = new ZipArchive;
				if ($zip->open($zip_file) === TRUE) {
					$zip->extractTo($demo_directory);
					$zip->close();

					// Remove the zip file
					unlink($zip_file);

					// Get the path to the extracted uploads folder
					$extracted_uploads_folder = $demo_directory . $folder_name . '/upload/';

					// Check if the uploads folder exists
					if (is_dir($extracted_uploads_folder)) {
						// Get list of all directories within the uploads folder
						$directories = glob($extracted_uploads_folder . '*', GLOB_ONLYDIR);

						// Move each directory to the WordPress uploads directory
						foreach ($directories as $directory) {
							$destination = WP_CONTENT_DIR . '/uploads/' . basename($directory);

							if (file_exists($destination)) {
								// Remove existing destination folder
								$this->delete_dir($destination);
							}

							// Move the directory
							rename($directory, $destination);

							// Replace URLs in XML files
							$xml_file = $demo_directory . ''.$folder_name.'/hostiko-'.preg_replace("/-/", "", $folder_name).'-content.xml';
							if (file_exists($xml_file)) {
								$xml_content = file_get_contents($xml_file);
								$current_site_url = site_url() . '/wp-content/uploads/';
								$xml_content = str_replace('https://hostiko.com/'.preg_replace("/-/", "", $folder_name).'/wp-content/uploads/', $current_site_url, $xml_content);
								// $xml_content = str_replace('https://hostiko.com/'.preg_replace("/-/", "", $folder_name).'/', site_url() . '/', $xml_content);

								file_put_contents($xml_file, $xml_content);
								
							} else {
								// echo "XML file $xml_file not found.";
							}
						}

						// Move the extracted folder to the current uploads directory
						$this->move_extracted_folder($demo_directory, $folder_name);
					} else {
						// echo "Failed to find the unzipped uploads folder.";
					}
				} else {
					// echo "Failed to extract zip file.";
				}
			}
		}

	}

	/**
	 * Move the extracted folder to the current uploads directory.
	 *
	 * @param string $demo_directory
	 * @param string $folder_name
	 */
	private function move_extracted_folder($demo_directory, $folder_name) {
		$source = $demo_directory . $folder_name;
		$upload_dir = wp_upload_dir();
		$destination = $upload_dir['path'] . '/' . $folder_name;

		if (is_dir($source)) {
			if (file_exists($destination)) {
				// Remove existing destination folder
				$this->delete_dir($destination);
			}
			rename($source, $destination);
		}
	}

	/**
	 * Method to remove GDPR cookie compliance warnings from the merlin page
	 */
	private function akd_remove_cookie_compliance_warnings(){
		//cookie complaince plugin warning on merlin page
		if ( is_plugin_active( 'gdpr-cookie-compliance/moove-gdpr.php' ) ) {
			if(isset($_GET['page']) && $_GET['page'] === 'merlin'){
				$this->remove_class_action('gdpr_check_review_banner_condition', 'Moove_GDPR_Review', 'gdpr_check_review_banner_condition_func');
			}
		}
	}

	/**
	 * Method to import demo content files
	 */
	public function akd_import_theme_demo_files() {

		$uploads_dir = wp_upload_dir()['path'];

		// Check if the directory exists
		if (is_dir($uploads_dir)) {
			// Scan the directory for subfolders matching the pattern 'layout-*'
			$subfolders = array_filter(glob($uploads_dir . '/layout-*'), 'is_dir');

			// Initialize an empty array to hold the layout data
			$layouts = array();

			// Loop through each found subfolder
			foreach ($subfolders as $folder) {
				// Extract the layout number from the folder name
				if (preg_match('/layout-(\d+)/', basename($folder), $matches)) {
					$layout_number = $matches[1];

					// Create the array entry for the current layout
					$layouts[] = array(
						'import_file_name' => 'Hostiko Layout ' . $layout_number,
						'categories' => array(),
						'local_import_file' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/hostiko-layout' . $layout_number . '-content.xml',
						'local_import_widget_file' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/site-settings.json',
						'local_import_customizer_file' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/hostiko-customizer.dat',
						'local_import_options_file' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/exported-options.json',
						'import_preview_image_url' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/screenshot.png',
					);
				}
			}
		} else {
			echo 'Uploads directory does not exist.';
		}

		return $layouts;

	}

	/**
	 * Method to remove specific action or filter
	 */
	private function remove_class_action ($action,$class,$method) {
		global $wp_filter ;
		if (isset($wp_filter[$action])) {
			$len = strlen($method) ;
			foreach ($wp_filter[$action] as $pri => $actions) {
				foreach ($actions as $name => $def) {
					if (substr($name,-$len) == $method) {
						if (is_array($def['function'])) {
							if (get_class($def['function'][0]) == $class) {
								if (is_object($wp_filter[$action]) && isset($wp_filter[$action]->callbacks)) {
									unset($wp_filter[$action]->callbacks[$pri][$name]) ;
								} else {
									unset($wp_filter[$action][$pri][$name]) ;
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Method to get elementor global site settings
	 */
	private function akd_get_elementor_global_settings(){
		// Check if Elementor plugin is active
		if (in_array('elementor/elementor.php', apply_filters('active_plugins', get_option('active_plugins')))) {
			if(isset($_POST['akd-framework-admin-page'])){
				if(isset($_POST['akd-framework-admin-page']['akd-theme-dashboard']) && is_array($_POST['akd-framework-admin-page']['akd-theme-dashboard'])){

					global $wpdb;

					$active_kit_id = get_option('elementor_active_kit');
					if($active_kit_id){
						$post_meta = get_post_meta($active_kit_id, '_elementor_page_settings', true);

						if($post_meta){
							$akd_theme_system_colors = $_POST['akd-framework-admin-page']['akd-theme-dashboard']['akd-theme-colors'];
							$akd_theme_custom_colors = $_POST['akd-framework-admin-page']['akd-theme-dashboard']['akd-custom-colors'];
							$akd_theme_typography = $_POST['akd-framework-admin-page']['akd-theme-dashboard'];
							
							//typography update
							$post_meta['system_typography'][0]['typography_font_family'] = $akd_theme_typography['akd-primary-typography']; 
							$post_meta['system_typography'][0]['typography_font_weight'] = $akd_theme_typography['akd-primary-font-weight']; 
							$post_meta['system_typography'][1]['typography_font_family'] = $akd_theme_typography['akd-secondary-typography'];
							$post_meta['system_typography'][1]['typography_font_weight'] = $akd_theme_typography['akd-secondary-font-weight']; 
							$post_meta['system_typography'][2]['typography_font_family'] = $akd_theme_typography['akd-text-typography'];
							$post_meta['system_typography'][2]['typography_font_weight'] = $akd_theme_typography['akd-text-font-weight'];
							
							// Update colors based on second array
							foreach ($post_meta['custom_colors'] as &$color) {
								// Extract the ID from the _id field
								$id = $color['_id'];
								
								// Check if the corresponding custom color exists
								$custom_color_key = 'custom-color-' . $id;
								if (array_key_exists($custom_color_key, $akd_theme_custom_colors)) {
									// Update the color value
									$color['color'] = $akd_theme_custom_colors[$custom_color_key];
								}
							}

							$post_meta['system_colors'][0]['color'] = $akd_theme_system_colors['primary'] ? $akd_theme_system_colors['primary'] : '';
							$post_meta['system_colors'][1]['color'] = $akd_theme_system_colors['secondary'] ? $akd_theme_system_colors['secondary'] : '';
							$post_meta['system_colors'][2]['color'] = $akd_theme_system_colors['text'] ? $akd_theme_system_colors['text'] : '';
							$post_meta['system_colors'][3]['color'] = $akd_theme_system_colors['accent'] ? $akd_theme_system_colors['accent'] : '';

							update_post_meta($active_kit_id, '_elementor_page_settings', $post_meta);

							$active_kit = \Elementor\Plugin::$instance->kits_manager->get_active_kit_for_frontend();
							if($active_kit){

								$query = $wpdb->prepare("
									SELECT *
									FROM $wpdb->postmeta
									WHERE 
										CONVERT(meta_id USING utf8) LIKE %s OR
										CONVERT(post_id USING utf8) LIKE %s OR
										CONVERT(meta_key USING utf8) LIKE %s OR
										CONVERT(meta_value USING utf8) LIKE %s
								", '%_elementor_page_settings%', '%_elementor_page_settings%', '%_elementor_page_settings%', '%_elementor_page_settings%');

								$results = $wpdb->get_results($query);

								// Handle results
								if ($results) {
									foreach ($results as $key => $result) {
										if($key == 0){
											$elementor_post_data = get_post_meta( $result->post_id, '_elementor_page_settings', true );
											$elementor_post_data['system_colors'][0]['color'] = $akd_theme_system_colors['primary'] ? $akd_theme_system_colors['primary'] : '';
											$elementor_post_data['system_colors'][1]['color'] = $akd_theme_system_colors['secondary'] ? $akd_theme_system_colors['secondary'] : '';
											$elementor_post_data['system_colors'][2]['color'] = $akd_theme_system_colors['text'] ? $akd_theme_system_colors['text'] : '';
											$elementor_post_data['system_colors'][3]['color'] = $akd_theme_system_colors['accent'] ? $akd_theme_system_colors['accent'] : '';
											// $elementor_post_data['custom_colors'][5]['color'] = '#dd3333';

											$all_settings = $active_kit->get_settings();
											$all_settings['system_colors'][0]['color'] = $akd_theme_system_colors['primary'] ? $akd_theme_system_colors['primary'] : '';
											$all_settings['system_colors'][1]['color'] = $akd_theme_system_colors['secondary'] ? $akd_theme_system_colors['secondary'] : '';
											$all_settings['system_colors'][2]['color'] = $akd_theme_system_colors['text'] ? $akd_theme_system_colors['text'] : '';
											$all_settings['system_colors'][3]['color'] = $akd_theme_system_colors['accent'] ? $akd_theme_system_colors['accent'] : '';
											// $all_settings['custom_colors'][5]['color'] = '#dd3333';
											// $all_settings['site_name'] = 'New Site Title';
											$active_kit->set_settings($all_settings);
											$active_kit->save($all_settings);

											// update_post_meta( 9057, '_elementor_page_settings', $elementor_post_data );
											$elementor_global_settings = json_encode($active_kit->get_settings(), true);
											update_option( '_akd_elementor_global', $elementor_global_settings );
										}
									}
								}
							}
						}
					}
				}
			}

			// $postMeta['custom_colors'][5]['color'] = '#dd3333';
			// Elementor is active, proceed to retrieve global theme settings
			// $elementor_theme_settings = \Elementor\Core\Settings\Manager::get_settings_managers('general')->get_model()->get_settings();
			
		}
	}

	/**
	 * Method to update elementor global site settings
	 */
	public function akd_update_elementor_global_settings($options){

	}

	public function checkCurl() {
		if(isset($_POST['action']) && trim($_POST['action']) == 'checkCurl'){

			if(get_option('_purchase_code_validate') == true){
				// Use the API credentials
				$apiUrl = $this->adjustUrl(trim($_POST['url']));
				$handles = curl_init($apiUrl);
				curl_setopt($handles, CURLOPT_NOBODY, true);
				curl_setopt($handles, CURLOPT_TIMEOUT, 10);
				curl_exec($handles);
				$resultat = curl_getinfo($handles, CURLINFO_HTTP_CODE);

				if($resultat){
						$_return['type'] = 'success';
						$_return['msg'] = 'SUCCESS: Curl Connected Successfully';
				}else{
					$_return['type'] = 'error';
					$_return['msg'] = 'Unable to connect to Server. Please contact to your hosting provider to enable your Curl';
				}
			} else {
				$_return['type'] = 'error';
				$_return['msg'] = 'Please activate your theme first.';
			}
		
		}

		echo json_encode( $_return );

		die();
	}

	private function adjustUrl($apiUrl) {
		$apiUrl = rtrim($apiUrl, '/');
		if (!preg_match('/includes/', $apiUrl)) {
			$apiUrl .= '/includes';
		}

		if (!preg_match('/api.php/', $apiUrl)) {
			$apiUrl .= '/api.php';
		}

		return $apiUrl;
	}

	public function getGroups() {
		$_return = array();
		if (isset($_POST['action']) && trim($_POST['action']) == 'getGroups') {
	  
			// Use the API credentials
			$apiUrl = $this->adjustUrl(trim($_POST['url']));
			$apiSecret = trim($_POST['secret']);
			$apiIdentifier = trim($_POST['identifier']);

			$_return = self::getData([
				'url' => $apiUrl,
				'identifier' => $apiIdentifier,
				'secret' => $apiSecret,
			]);

		}
		
		echo json_encode( $_return );
		die();
	}

	public function getData($configs) {
		$apiUrl = $configs['url'];
		$apiIdentifier = $configs['identifier'];
		$apiSecret = $configs['secret'];

		// Initialize the cURL resource
		$ch = $this->initCurl($apiUrl);
	
		// Call WHMCS API to get all products
		$params = array(
			'identifier' => $apiIdentifier,
			'secret' => $apiSecret,
			'action' => 'GetProducts',
			'responsetype' => 'json'
		);

		$responseData = $this->callApi($ch, $params);

		// Close the cURL resource
		$this->closeCurl($ch);
	
		$_return = array();
		$html = '';
		if ($responseData['result'] == 'success') {
			 //Save WHMCS data
			 update_option('whmcs_url', $apiUrl);
			 update_option('whmcs_secret', $this->encrypt_data($apiSecret, 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9'));
			 update_option('whmcs_identifier', $this->encrypt_data($apiIdentifier, 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9'));
 
			if(isset($responseData['products'])){
				update_option('whmcs_products_data', json_encode($responseData));

				$_return['type'] = 'success';
				$_return['data'] = $responseData;
				$_return['msg'] = 'SUCCESS: WHMCS data synced successfully.';
			} else {
				$_return['type'] = 'success';
				$_return['data'] = $responseData;
				$_return['msg'] = 'SUCCESS: WHMCS data synced successfully, No products Found';
			}
		} else {
			$_return['type'] = 'error';
			$_return['data'] = $responseData;
			$_return['msg'] = 'ERROR: '.$responseData['message'];
		}

		return $_return;
	}

	public function initCurl($apiUrl) {
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $apiUrl);
		curl_setopt($ch, CURLOPT_POST, true);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		return $ch;
	}

	public function callApi($ch, $params) {
		curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
		$response = curl_exec($ch);
		if (curl_errno($ch)) {
			$error = curl_error($ch);
			echo 'CURL Error: ' . $error;
		} else {
			$responseData = json_decode($response, true);
			return $responseData;
		}
	}

	public function closeCurl($ch) {
		curl_close($ch);
	}

	public function encrypt_data($data, $key) {
		$cipher = "aes-256-cbc";
		$iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($cipher));
		$encrypted = openssl_encrypt($data, $cipher, $key, 0, $iv);
		return base64_encode($iv . $encrypted);
	}

	public function getTLDs(){
		$apiUrl = $this->adjustUrl(trim($_POST['url']));
		$apiSecret = trim($_POST['secret']);
		$apiIdentifier = trim($_POST['identifier']);

		$ch = $this->initCurl($apiUrl);

		// Call WHMCS API to get TLD pricing
		$params = array(
			'identifier' => $apiIdentifier,
			'secret' => $apiSecret,
			'action' => 'GetTLDPricing',
			'responsetype' => 'json'
		);

		$responseData = $this->callApi($ch, $params);
		$this->closeCurl($ch);
		$_return = array();
		if($responseData['result'] == 'success'){
			if(isset($responseData['pricing'])){
				update_option('whmcs_products_tld_data', json_encode($responseData));
				$_return['type'] = 'success';
				$_return['data'] = json_encode($responseData);
				$_return['msg'] = 'SUCCESS: WHMCS TLD data synced successfully.';
			}
		}
		else{
			$_return['type'] = 'error';
			$_return['data'] = json_encode($responseData);
			$_return['msg'] = 'ERROR: '.$responseData['message'];
		}

		echo json_encode( $_return );
		die();
	}


	public function getGroups_resync() {
		$_return = [];
		if (isset($_POST['action']) && trim($_POST['action']) == 'getGroups_resync') {
			$apiUrl = $this->adjustUrl(trim($_POST['url']));

			$tempSecret = trim($_POST['secret']);
			$tempIdentifier = trim($_POST['identifier']);

			$apiSecret = decrypt_data($tempSecret, 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9');
			$apiIdentifier = decrypt_data($tempIdentifier , 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9');
			$apiSecret = $apiSecret == "" ? $tempSecret : $apiSecret;
			$apiIdentifier = $apiIdentifier == "" ? $tempIdentifier : $apiIdentifier;
			
			$configs = [
				'url' => $apiUrl,
				'identifier' => $apiIdentifier,
				'secret' => $apiSecret
			];
			
			$_return = self::getData($configs);
		}

		echo json_encode( $_return );
		die();
	}

	private function decrypt_data($data, $key) {
		$cipher = "aes-256-cbc";
		$data = base64_decode($data);
		$iv_length = openssl_cipher_iv_length($cipher);
		$iv = substr($data, 0, $iv_length);
		$data = substr($data, $iv_length);
		return openssl_decrypt($data, $cipher, $key, 0, $iv);
	}
}
